// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


package com.vasco.image.sample;

import com.vasco.image.exception.ImageGeneratorSDKException;
import com.vasco.image.generator.BWQRCodeResponse;
import com.vasco.image.generator.ImageGeneratorSDK;
import com.vasco.image.generator.ImageGeneratorSDKConstants;

import java.awt.image.BufferedImage;
import java.io.File;
import java.util.logging.Logger;

import javax.imageio.ImageIO;

/** Sample class for the ImageGenerator */
public class ImageGeneratorSDKSample {

    // Number of pixels per square in Cronto Image
    private static final int CRONTO_IMAGE_SQUARE_SIZE = 6;

    // If the image is to be printed on paper
    private static final boolean CRONTO_IMAGE_ON_PAPER = true;

    // Size in pixels of the generated QRCode image
    private static final int QRCODE_IMAGE_SIZE = 210;

    // File name of the generated Cronto Image
    private static final String CRONTO_IMAGE_FILENAME = "crontoImage.png";

    // File name of the generated QRCode image
    private static final String QRCODE_FILENAME = "qrCode.png";

    // Test string to generate the Cronto Image
    private static final String CRONTO_TEST_STRING =
            "5465737420537472696e6720666f7220496d61676547656e657261746f72";

    // Test string to generate the QRCode image
    private static final String QRCODE_TEST_STRING = "Test String for ImageGenerator";

    private static final Logger logger =
            Logger.getLogger(ImageGeneratorSDKSample.class.getSimpleName());

    public static void main(String[] args) {

        logger.info("*************************************************************************\n");
        logger.info("*               Image Generator - Sample demonstration                  *\n");
        logger.info(
                "*************************************************************************\n\n");

        // Generate Cronto image
        demoGenerateDynamicCrontoImage();

        // Generate QR Code image
        demoGenerateQRCode();
    }

    /**
     * Example of how to generate a Cronto image Generate a Cronto image from the "Test String for
     * ImageGenerator" String in its hexadecimal representation.
     *
     * <p>The CRONTO images generated with the GenerateDynamicCrontoImage method cannot be scanned
     * by VASCO hardware authenticators (e.g. DP760, DP780).
     *
     * <p>Use the GenerateCrontoSign method to generate CRONTO images compatible with VASCO hardware
     * authenticators (e.g. DP760, DP780).
     */
    private static void demoGenerateDynamicCrontoImage() {
        try {
            logger.info("Generating Cronto image...");

            // Generate Cronto image
            BufferedImage crontoImg =
                    ImageGeneratorSDK.generateDynamicCrontoImage(
                            CRONTO_IMAGE_SQUARE_SIZE, CRONTO_TEST_STRING, CRONTO_IMAGE_ON_PAPER);

            // Write output file
            File outFile = new File(CRONTO_IMAGE_FILENAME);
            ImageIO.write(crontoImg, "png", outFile);

            logger.info("Cronto image generated : " + CRONTO_IMAGE_FILENAME);
        } catch (ImageGeneratorSDKException e) {
            logger.severe("Cronto image generation failed: " + e.getErrorMessage());
        } catch (Exception e) {
            logger.severe("Cronto image generation failed: " + e.getMessage());
        }
        logger.info("\n");
    }

    /**
     * Example of how to generate a QR Code. Generate a QR Code from the "Test String for
     * ImageGenerator" String.
     */
    private static void demoGenerateQRCode() {
        try {
            logger.info("Generating QR Code image...");

            // Generate QR Code
            BWQRCodeResponse qrCodeResponse =
                    ImageGeneratorSDK.generateBWQRCode(
                            QRCODE_IMAGE_SIZE,
                            QRCODE_TEST_STRING,
                            ImageGeneratorSDKConstants.QRCODE_ERROR_CORRECTION_LEVEL_LOW);
            BufferedImage qrCodeImg = qrCodeResponse.getImage();

            // Write output file
            File outFile = new File(QRCODE_FILENAME);
            ImageIO.write(qrCodeImg, "png", outFile);

            logger.info("QR Code image generated : " + QRCODE_FILENAME);
        } catch (ImageGeneratorSDKException e) {
            logger.severe("QR Code image generation failed: " + e.getErrorMessage());
        } catch (Exception e) {
            logger.severe("QR Code image generation failed: " + e.getMessage());
        }
        logger.info("\n");
    }
}
